using System;
using System.Collections.Generic;
using System.ComponentModel;
using System.IO;
using System.Drawing;
using System.Runtime.InteropServices;
using System.Threading;
using System.Text;
using System.Windows.Forms;
using DynaPDF;

namespace renderpage
{
   public partial class Form1 : Form
   {
      [StructLayout(LayoutKind.Sequential, CharSet=CharSet.Ansi, Pack = 0)]
      class RECT
      {
         public int left;
         public int top;
         public int right;
         public int bottom;
         public RECT(Rectangle r)
         {
            left   = r.Left;
            top    = r.Top;
            right  = r.Right;
            bottom = r.Bottom;
         }
      }
      // ExtTextOut flag
      const int ETO_OPAQUE     = 0x00000002;

      // Window style flags
      const int WS_CAPTION     = 0x00C00000;
      const int WS_SYSMENU     = 0x00080000;
      const int WS_THICKFRAME  = 0x00040000;
      const int WS_MINIMIZEBOX = 0x00020000;

      // Class style flags
      const int CS_VREDRAW     = 0x0001;
      const int CS_HREDRAW     = 0x0002;
      const int CS_OWNDC       = 0x0020;

      const uint APP_BACK_COLOR    = 0x00505050;
      const int APP_CLIENT_BORDER  = 6;
      const int APP_CLIENT_BORDER2 = 3;

      private bool            m_AdjWindow;
      private int             m_BorderX;
      private int             m_BorderY;
      private int             m_CurrPage;
      private IntPtr          m_CurrPageObj;
      private IntPtr          m_DC;
      private int             m_DestX;
      private int             m_DestY;
      private int             m_ImgH;
      private int             m_ImgW;
      private byte[]          m_ImpPages;
      private int             m_PageCount;
      static private CPDF     m_PDF;
      private IntPtr          m_RAS;
      private TPDFRasterImage m_RasImage;
      private Thread          m_RenderThread;
      private int             m_ScreenH;
      private int             m_ScreenW;
      private bool            m_Update;

      [DllImport("gdi32.dll", CharSet=CharSet.Auto)]    private static extern int    ExtTextOut(IntPtr hdc, int X, int Y, int fuOptions, RECT lprc, IntPtr lpString, int cbCount, IntPtr lpDx);
      [DllImport("user32.dll", CharSet=CharSet.Ansi)]   private static extern IntPtr GetDC(IntPtr hWnd);
      [DllImport("gdi32.dll", CharSet=CharSet.Unicode)] private static extern int    GetICMProfileW(IntPtr hdc, ref int Size, StringBuilder Filename);
      [DllImport("user32.dll", CharSet=CharSet.Ansi)]   private static extern int    ReleaseDC(IntPtr hWnd, IntPtr hdc);
      [DllImport("gdi32.dll", CharSet=CharSet.Ansi)]    private static extern int    SetBkColor(IntPtr hdc, uint crColor);
      [DllImport("user32.dll", SetLastError = true)]    private static extern int    MoveWindow(IntPtr hWnd, int X, int Y, int nWidth, int nHeight, int bRepaint);

      int PDFError(IntPtr Data, int ErrCode, IntPtr ErrMessage, int ErrType)
      {
         MessageBox.Show(System.Runtime.InteropServices.Marshal.PtrToStringAnsi(ErrMessage));
         return 0;
      }

      public Form1()
      {
         InitializeComponent();
         // Required flags to avoid flickering
         this.SetStyle(ControlStyles.UserPaint | ControlStyles.AllPaintingInWmPaint | ControlStyles.ResizeRedraw | ControlStyles.Opaque, true);
         this.UpdateStyles();

         // Initialize the TRasterImage structure
         m_RasImage                       = new TPDFRasterImage();
         m_RasImage.StructSize            = Marshal.SizeOf(m_RasImage);
         m_RasImage.DefScale              = TPDFPageScale.psFitBest;
         m_RasImage.InitWhite             = 1;
         // We draw the image with SetDIBitsToDevice() and this function does not support alpha transparency.
         // To get a correct result we pre-blend the image with a white background.
         m_RasImage.Flags                 = TRasterFlags.rfDefault | TRasterFlags.rfCompositeWhite;
         m_RasImage.Matrix.a              = 1.0;
         m_RasImage.Matrix.d              = 1.0;
         m_RasImage.UpdateOnImageCoverage = 0.5f;
         m_RasImage.UpdateOnPathCount     = 1000;

         m_DestX   = APP_CLIENT_BORDER2;
         m_DestY   = APP_CLIENT_BORDER2;
         m_ScreenW = Screen.PrimaryScreen.Bounds.Width;
         m_ScreenH = Screen.PrimaryScreen.Bounds.Height;
         m_BorderX = Width  - this.ClientRectangle.Width  + APP_CLIENT_BORDER;
         m_BorderY = Height - this.ClientRectangle.Height + APP_CLIENT_BORDER;
      }

      protected override CreateParams CreateParams
      {
         get
         {
            CreateParams cp = base.CreateParams;
            // We need a private dc! The flag CS_OWNDC is required.
            cp.Style      = WS_CAPTION | WS_SYSMENU | WS_MINIMIZEBOX | WS_THICKFRAME;
            cp.ClassStyle = CS_OWNDC | CS_VREDRAW | CS_HREDRAW;
            return cp;
         }
      }

      void AddPage(int PageNum)
      {
         m_ImpPages[PageNum>>3] |= (byte)(0x80 >> (PageNum & 7));
      }

      private void Form1_FormClosed(object sender, FormClosedEventArgs e)
      {
         StopRenderThread();
         if (m_RAS != IntPtr.Zero) m_PDF.DeleteRasterizer(ref m_RAS);
         m_PDF = null;
      }

      private void Form1_KeyDown(object sender, KeyEventArgs e)
      {
         if (e.Control && e.KeyCode == Keys.O)
            OpenFile();
         else
         {
            switch (e.KeyCode)
            {
               case Keys.Up:   RenderNextPage(m_CurrPage-1); break;
               case Keys.Down: RenderNextPage(m_CurrPage+1); break;
            }
         }
      }

      private void Form1_Load(object sender, EventArgs e)
      {
         m_DC  = GetDC(this.Handle);
         m_PDF = new CPDF();
         // Get the monitor profile
         StringBuilder str = new StringBuilder(512);
         int size = str.Capacity -1;
         IntPtr dc = GetDC(IntPtr.Zero);
            GetICMProfileW(dc, ref size, str);
         ReleaseDC(IntPtr.Zero, dc);

         m_PDF.SetOnErrorProc(IntPtr.Zero, new TErrorProc(PDFError));
         // It is important to set an absolute path here since a relative path
         // doesn't work if the working directory will be changed at runtime.
         // The flag lcmDelayed makes sure that the cmaps will only be loaded
         // if necessary.
         m_PDF.SetCMapDir(Path.GetFullPath("../../../../../../../Resource/CMap/"), TLoadCMapFlags.lcmRecursive | TLoadCMapFlags.lcmDelayed);
         // Initialize color management
         TPDFColorProfiles p = new TPDFColorProfiles();
         p.StructSize     = Marshal.SizeOf(p);
         p.DefInCMYKW     = Path.GetFullPath("../../../../../../test_files/ISOcoated_v2_bas.ICC");
         p.DeviceProfileW = str.ToString();
         m_PDF.InitColorManagement(p, TPDFColorSpace.csDeviceRGB, TPDFInitCMFlags.icmBPCompensation | TPDFInitCMFlags.icmCheckBlackPoint);
         str = null;

         m_RAS = m_PDF.CreateRasterizerEx(m_DC, ClientRectangle.Width - APP_CLIENT_BORDER, ClientRectangle.Height - APP_CLIENT_BORDER, TPDFPixFormat.pxfBGRA);
      }

      private void Form1_MouseWheel(object sender, MouseEventArgs e)
      {
         if (e.Delta < 0)
            RenderNextPage(m_CurrPage+1);
         else
            RenderNextPage(m_CurrPage-1);
      }

      private void Form1_Paint(object sender, PaintEventArgs e)
      {
/*
         SolidBrush b = new SolidBrush(Color.FromArgb((int)APP_BACK_COLOR));
         e.Graphics.FillRectangle(b, 0, 0, APP_CLIENT_BORDER2, ClientRectangle.Height);
         e.Graphics.FillRectangle(b, APP_CLIENT_BORDER2, ClientRectangle.Bottom-APP_CLIENT_BORDER2, ClientRectangle.Right-APP_CLIENT_BORDER2, APP_CLIENT_BORDER2);
         e.Graphics.FillRectangle(b, ClientRectangle.Right-APP_CLIENT_BORDER2, ClientRectangle.Top, APP_CLIENT_BORDER2, ClientRectangle.Height);
         e.Graphics.FillRectangle(b, APP_CLIENT_BORDER2, ClientRectangle.Top, ClientRectangle.Right-APP_CLIENT_BORDER2, APP_CLIENT_BORDER2);
*/
         // Plain GDI code is faster as the new GDI+ stuff
         RECT r = new RECT(ClientRectangle);
         SetBkColor(m_DC, APP_BACK_COLOR);
         // Left
         r.left   = 0;
         r.right  = APP_CLIENT_BORDER2;
         r.bottom = ClientRectangle.Bottom;
         r.top    = ClientRectangle.Top;
         ExtTextOut(m_DC, 0, 0, ETO_OPAQUE, r, IntPtr.Zero, 0, IntPtr.Zero);
         // Bottom
         r.left   = APP_CLIENT_BORDER2;
         r.right  = ClientRectangle.Right;
         r.bottom = ClientRectangle.Bottom;
         r.top    = ClientRectangle.Bottom - APP_CLIENT_BORDER2;
         ExtTextOut(m_DC, 0, 0, ETO_OPAQUE, r, IntPtr.Zero, 0, IntPtr.Zero);
         // Right
         r.left   = APP_CLIENT_BORDER2;
         r.right  = ClientRectangle.Right;
         r.bottom = ClientRectangle.Top + APP_CLIENT_BORDER2;
         r.top    = ClientRectangle.Top;
         // Top
         ExtTextOut(m_DC, 0, 0, ETO_OPAQUE, r, IntPtr.Zero, 0, IntPtr.Zero);
         r.left   = ClientRectangle.Right - APP_CLIENT_BORDER2;
         r.right  = ClientRectangle.Right;
         r.bottom = ClientRectangle.Bottom;
         r.top    = ClientRectangle.Top;
         ExtTextOut(m_DC, 0, 0, ETO_OPAQUE, r, IntPtr.Zero, 0, IntPtr.Zero);
         if (m_Update)
         {
            if (m_RenderThread == null) StartRenderThread();
         }else if (m_CurrPage > 0)
            m_PDF.Redraw(m_RAS, m_DC, m_DestX, m_DestY);
      }

      private void Form1_Resize(object sender, EventArgs e)
      {
         RECT r = new RECT(ClientRectangle);
         SetBkColor(m_DC, APP_BACK_COLOR);
         ExtTextOut(m_DC, 0, 0, ETO_OPAQUE, r, IntPtr.Zero, 0, IntPtr.Zero);
         if (m_RAS != IntPtr.Zero && m_PDF.ResizeBitmap(m_RAS, m_DC, m_ImgW, m_ImgH))
         {
            m_AdjWindow = true;
            RenderCurrPage();
         }
      }

      private void Form1_Shown(object sender, EventArgs e)
      {
         OpenFile();
      }

      bool IsPageAvailable(int PageNum)
      {
         return (m_ImpPages[PageNum>>3] & (0x80 >> (PageNum & 7))) != 0;
      }

      private void OpenFile()
      {
         this.Text = "RenderPage()";
         RECT r = new RECT(ClientRectangle);
         SetBkColor(m_DC, APP_BACK_COLOR);
         ExtTextOut(m_DC, 0, 0, ETO_OPAQUE, r, IntPtr.Zero, 0, IntPtr.Zero);
         m_CurrPage  = 0;
         m_PageCount = 0;
         StopRenderThread();
         if (openFileDialog.ShowDialog() == DialogResult.OK)
         {
            this.Text = openFileDialog.FileName;
            if (m_PDF.HaveOpenDoc()) m_PDF.FreePDF();
            m_PDF.CreateNewPDF(null); // We create no PDF file in this example

            if (m_PDF.OpenImportFile(openFileDialog.FileName, TPwdType.ptOpen, null) < 0) return;

            // We import pages manually in this example and therefore, no global objects will
            // be imported as it would be the case if ImportPDFFile() would be used.
            // However, the one and only thing we need is the output intent for correct
            // color management. Anything else can be discarded.
            m_PDF.SetImportFlags(TImportFlags.ifContentOnly);
            m_PDF.ImportCatalogObjects();

            m_PDF.SetImportFlags(TImportFlags.ifImportAll | TImportFlags.ifImportAsPage); // The flag ifImportAsPage makes sure that pages will not be converted to templates.
            m_PDF.SetImportFlags2(TImportFlags2.if2UseProxy);                             // The flag if2UseProxy reduces the memory usage.

            m_PageCount = m_PDF.GetInPageCount();
            m_ImpPages  = new byte[(m_PageCount >> 3) + 1];

            m_AdjWindow = true;
            m_CurrPage  = 1;
            RenderCurrPage();
         }
      }

      void RenderCurrPage()
      {
         StopRenderThread();
         if (m_CurrPage > 0)
         {
            if (!IsPageAvailable(m_CurrPage))
            {
               // No need to check the return value of ImportPageEx(). Nothing critical happens
               // if the function fails. We get just an empty page in this case.
               m_PDF.EditPage(m_CurrPage);
                  m_PDF.ImportPageEx(m_CurrPage, 1.0, 1.0);
               m_PDF.EndPage();
               AddPage(m_CurrPage);
            }
            // This check is required to avoid critical errors.
            if ((m_CurrPageObj = m_PDF.GetPageObject(m_CurrPage)) == IntPtr.Zero) return;
            m_PDF.CalcPagePixelSize(m_CurrPageObj, TPDFPageScale.psFitBest, 1.0f, m_ScreenW-m_BorderX, m_ScreenH-m_BorderY, m_RasImage.Flags, ref m_ImgW, ref m_ImgH);
            if (m_AdjWindow)
            {
               int w = m_ImgW + m_BorderX;
               int h = m_ImgH + m_BorderY;
               if (ClientRectangle.Width - APP_CLIENT_BORDER != m_ImgW || ClientRectangle.Height - APP_CLIENT_BORDER != m_ImgH)
               {
                  m_AdjWindow = false;
                  MoveWindow(this.Handle, (m_ScreenW-w) >> 1, (m_ScreenH-h) >> 1, w, h, 1);
               }
            }
            m_Update = true;
            Invalidate();
         }
      }

      void RenderNextPage(int PageNum)
      {
         if (PageNum < 1 || PageNum > m_PageCount || PageNum == m_CurrPage) return;
         m_CurrPage  = PageNum;
         m_AdjWindow = true;
         RenderCurrPage();
      }

      void RenderPageFunc()
      {
         m_PDF.RenderPageEx(m_DC, ref m_DestX, ref m_DestY, m_CurrPageObj, m_RAS, ref m_RasImage);
         m_Update = false;
      }

      void StartRenderThread()
      {
         StopRenderThread();
         m_Update = false;
         m_RenderThread = new Thread(RenderPageFunc);
         m_RenderThread.Priority = ThreadPriority.Lowest;
         m_RenderThread.Start();
      }

      void StopRenderThread()
      {
         if (m_RenderThread != null)
         {
            m_PDF.Abort(m_RAS);
            m_RenderThread.Join();
            m_RenderThread = null;
         }
      }
   }
}